<?php

namespace Noptin\Addons_Pack\Tasks;

/**
 * Contains the main task manager class.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main component Class.
 *
 */
class Main {

	/**
	 * Stores the main component instance.
	 *
	 * @access private
	 * @var    Main $instance The main component instance.
	 * @since  1.0.0
	 */
	private static $instance = null;

	/**
	 * Current task.
	 *
	 * @var Task
	 * @since 1.0.0
	 */
	private static $current_task = null;

	/**
	 * Scheduled tasks in this request.
	 *
	 * @var string[]
	 */
	private static $scheduled_tasks = array();

	/**
	 * The task runner.
	 *
	 * @var Task_Runner
	 */
	public $task_runner = null;

	/**
	 * Get active instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return Main The main component instance.
	 */
	public static function instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		$this->load_data_store();
		$this->task_runner = new Task_Runner();

		add_action( 'shutdown', array( __CLASS__, 'handle_unexpected_shutdown' ) );
		add_action( 'noptin_tasks_before_execute', array( __CLASS__, 'set_task' ), 0, 1 );
		add_action( 'noptin_tasks_after_execute', array( __CLASS__, 'reset_task' ), 0 );
		add_action( 'noptin_tasks_failed_execution', array( __CLASS__, 'reset_task' ), 0 );
		add_action( 'noptin_tasks_schedule', array( __CLASS__, 'schedule_task' ), 10, 3 );
		add_filter( 'get_noptin_admin_tools', array( $this, 'filter_admin_tools' ) );
		add_action( 'noptin_admin_tool_scheduled_tasks', array( $this, 'display_scheduled_tasks' ) );
	}

	/**
	 * Loads the data store.
	 */
	private function load_data_store() {
		\Hizzle\Store\Store::init( 'noptin', include plugin_dir_path( __FILE__ ) . 'db-schema.php' );
	}

	/**
	 * Returns the task statuses.
	 *
	 * @return array
	 */
	public static function get_statuses() {
		return array(
			'pending'  => __( 'Pending', 'noptin-addons-pack' ),
			'running'  => __( 'Running', 'noptin-addons-pack' ),
			'failed'   => __( 'Failed', 'noptin-addons-pack' ),
			'canceled' => __( 'Canceled', 'noptin-addons-pack' ),
			'complete' => __( 'Complete', 'noptin-addons-pack' ),
		);
	}

	public static function set_task( $task ) {
		self::$current_task = $task;
	}

	public static function reset_task() {
		self::$current_task = null;
	}

	public static function handle_unexpected_shutdown() {
		$error = error_get_last();

		if ( ! empty( $error ) && in_array( $error['type'], array( E_ERROR, E_PARSE, E_COMPILE_ERROR, E_USER_ERROR, E_RECOVERABLE_ERROR ), true ) ) {
			$task = self::$current_task;
			if ( ! empty( $task ) ) {
				$task->unexpected_shutdown( $task, $error );
			}
		}
	}

	/**
	 * Fetches tasks.
	 *
	 * @param array $args Query arguments.
	 * @param string $return 'results' returns the found records, 'count' returns the total count, 'aggregate' runs an aggregate query, while 'query' returns query object.
	 * @return int|array|Store\Record[]|Store\Query|WP_Error
	 */
	public static function query( $args = array(), $return = 'results' ) {

		// Do not retrieve all fields if we just want the count.
		if ( 'count' === $return ) {
			$args['fields'] = 'id';
			$args['number'] = 1;
		}

		// Do not count all matches if we just want the results.
		if ( 'results' === $return ) {
			$args['count_total'] = false;
		}

		try {
			$collection = \Hizzle\Store\Collection::instance( 'noptin_tasks' );
			$query      = $collection->query( $args );

			if ( 'results' === $return ) {
				return $query->get_results();
			}

			if ( 'count' === $return ) {
				return $query->get_total();
			}

			if ( 'aggregate' === $return ) {
				return $query->get_aggregate();
			}

			return $query;

		} catch ( \Hizzle\Store\Store_Exception $e ) {
			return new \WP_Error( $e->getErrorCode(), $e->getMessage(), $e->getErrorData() );
		}

	}

	/**
	 * Schedules a task to run in the background.
	 *
	 * @param string $hook
	 * @param array  $args
	 * @param int    $delay In seconds.
	 * @return false|\WP_Error|Task
	 */
	public static function schedule_task( $hook, $args = array(), $delay = 0 ) {

		$runs_on      = empty( $delay ) ? time() : strtotime( " +{$delay} seconds" );
		$args_hash    = md5( maybe_serialize( $args ) );
		$unique_check = $hook . '|' . $args_hash . '|' . $delay;

		// Ensure the same task is not scheduled twice in the same request.
		if ( in_array( $unique_check, self::$scheduled_tasks, true ) ) {
			return;
		}

		try {
			$collection = \Hizzle\Store\Collection::instance( 'noptin_tasks' );
			$task       = $collection->get( 0 );

			// Set the props.
			/** @var Task $task */
			$task->set_hook( $hook );
			$task->set_status( 'pending' );
			$task->set_args( wp_json_encode( $args ) );
			$task->set_args_hash( $args_hash );
			$task->set_date_scheduled( $runs_on );
			$task->save();

			self::$scheduled_tasks[] = $unique_check;

			return $task;
		} catch ( \Hizzle\Store\Store_Exception $e ) {
			return new \WP_Error( $e->getErrorCode(), $e->getMessage(), $e->getErrorData() );
		}
	}

	/**
	 * Deletes a scheduled task.
	 *
	 * @param string $hook
	 * @param array  $args
	 * @param int    $limit
	 * @return false|\WP_Error|Task
	 */
	public static function delete_scheduled_task( $hook, $args = array(), $limit = 1 ) {

		/** @var Task[] $tasks */
		$tasks = self::query(
			array(
				'status' => 'pending',
				'number' => $limit,
				'hook'   => $hook,
				'args'   => wp_json_encode( $args ),
			)
		);

		foreach ( $tasks as $task ) {
			$task->delete();
		}

	}

	public static function delete_old_tasks() {
		$lifespan = apply_filters( 'noptin_tasks_retention_period', MONTH_IN_SECONDS );

		/** @var Task[] $tasks */
		$tasks = self::query(
			array(
				'status'               => array( 'complete', 'canceled' ),
				'number'               => static::get_batch_size(),
				'date_modified_before' => gmdate( 'Y-m-d H:i:s', time() - $lifespan ),
			)
		);

		foreach ( $tasks as $task ) {
			$task->delete();
		}

	}

	/**
	 * Mark tasks that have been running for more than a given time limit as failed, based on
	 * the assumption some uncatachable and unloggable fatal error occurred during processing.
	 *
	 *
	 * @param int $time_limit The number of seconds to allow a task to run before it is considered to have failed. Default 300 (5 minutes).
	 */
	public static function mark_failures( $time_limit = 300 ) {
		$timeout = apply_filters( 'noptin_tasks_failure_period', $time_limit );
		if ( $timeout < 0 ) {
			return;
		}

		$tasks = self::query(
			array(
				'status'               => 'running',
				'number'               => static::get_batch_size(),
				'date_modified_before' => gmdate( 'Y-m-d H:i:s', time() - $timeout ),
			)
		);

		foreach ( $tasks as $task ) {
			$task->timed_out( $task, $timeout );
		}
	}

	/**
	 * Do all of the cleaning actions.
	 *
	 * @param int $time_limit The number of seconds to use as the timeout and failure period. Default 300 (5 minutes).
	 */
	public static function clean( $time_limit = 300 ) {
		static::delete_old_tasks();
		static::mark_failures( $time_limit );
	}

	/**
	 * Get the batch size for cleaning the queue.
	 *
	 * @return int
	 */
	protected static function get_batch_size() {
		return absint( apply_filters( 'noptin_tasks_cleanup_batch_size', 20 ) );
	}

	/**
	 * Filters admin tools.
	 *
	 * @param array $tools
	 * @return array
	 */
	public function filter_admin_tools( $tools ) {

		$tools['scheduled_tasks'] = array(
			'name'   => __( 'Scheduled Tasks', 'noptin-addons-pack' ),
			'button' => __( 'View', 'noptin-addons-pack' ),
			'desc'   => __( 'View a list of scheduled tasks.', 'noptin-addons-pack' ),
		);

		return $tools;
	}

	/**
	 * Displays the scheduled tasks tool.
	 */
	public function display_scheduled_tasks() {
		echo '<form method="post">';
		$table = new Tasks_Table();
		$table->prepare_items();
		$table->display();
		echo '</form>';
	}
}
